/*
 * Create a wiring list corresponding to the map file
 */
#include <unistd.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>

#include "enclosure.h"
#include "switchstuff.h"
#include "map.h"
#include "gmstuff.h"

extern int My_host_id;
extern char *Wirelist;
extern char Wiredir[256];
extern int Hport;

#define FMA_ENCLOSURES_FN         "enclosures.csv"
#define FMA_ENCLOSURES_TYPES_STR  "string,string"
#define FMA_ENCLOSURES_LABELS_STR "name,product_id"

#define FMA_HOSTS_FN              "hosts.csv"
#define FMA_HOSTS_TYPES_STR       "string,string"
#define FMA_HOSTS_LABELS_STR      "hostname,sw_version"

#define FMA_LINECARDS_FN           "linecards.csv"
#define FMA_LINECARDS_TYPES_STR    "string,integer,string,string"
#define FMA_LINECARDS_LABELS_STR   "enclosure_name,enclosure_slot,product_id,serial_no"

#define FMA_LINKS_FN              "links.csv"
#define FMA_LINKS_TYPES_STR       "string,integer,integer,integer,string,integer,integer,integer"
#define FMA_LINKS_LABELS_STR      "name_1,slot_1,port_1,subport_1,name_2,slot_2,port_2,subport_2"

#define FMA_NICS_FN               "nics.csv"
#define FMA_NICS_TYPES_STR        "string,integer,MAC,,string,string"
#define FMA_NICS_LABELS_STR       "hostname,host_nic_id,mac_addr,serial_no,product_id"


int
find_the_packets(
  struct enclosure **pkt_ep,
  int *pkt_slot,
  int *pkt_port,
  int min,
  int max)
{
  int sw;
  int slot;
  int p;
  int diff;
  struct enclosure *ep;
  struct slotdesc *sp;
  struct xbarport *xp;
  struct enclosure *save_ep;
  int save_slot;
  int save_port;

  /* parse all enclosure files */
  parse_all_enclosures();
  
  save_ep = NULL;	/* no match found yet */

  for (sw=0; sw < Nswitch; ++sw) {
    ep = Switch + sw;
    if (ep->unmapped_slots == 0) continue;	/* skip known enclosures */

    for (slot=0; slot<MAX_SLOT; ++slot) {
      sp = ep->slot[slot];
      if (sp == NULL || (sp->xbar != NULL)) continue;

      for (p=0; p<sp->num_xp; ++p) {
        xp = sp->xp + p;

	if (xp->vals.controlforxbarport != 0) continue;

        diff = xp->vals.invalidroutes - xp->old_vals.invalidroutes;
        if (diff > 0) {
          printf("%s, slot %d, port %d, diff=%d, o=%d, n=%d\n",
		ep->name, slot, p, diff,
		xp->old_vals.invalidroutes,
		xp->vals.invalidroutes);
        }
        if (diff >= min && diff <= max) {
          printf("I am attached to %s, slot %d, port %d\n",
		ep->name, slot, p);
	  if (save_ep == NULL) {
	    save_ep = ep;
	    save_slot = slot;
	    save_port = p;
          } else {
	    fprintf(stderr, "Multiple invalidroute matches\n");
	    return 0;
          }
        }
      }
    }
  }

  if (save_ep != NULL) {
    *pkt_ep = save_ep;
    *pkt_slot = save_slot;
    *pkt_port = save_port;
    return 1;
  }

  fprintf(stderr, "Ack! Cannot find packets in enclosure!\n");
  return 0;
}

int
create_wiring_list(struct mapfile *mp)
{
  struct enclosure *ep;
  struct enclosure *nep;
  struct slotdesc *sp;
  struct slotdesc *sp2;
  struct fiberport *fp;
  struct xbarport *xp;
  struct xbar *xbar;
  struct xbar *xbar2;
  int xport;
  int xport2;
  int s;
  int slot;
  int p;
  int rc;
  int nslot;
  int nport;
  int retry;

  /* get the xbar attached to our host */
  xbar = mp->host[My_host_id].port[Hport].xbar;
  
  for (retry=0; retry<30; ++retry) {
    send_invalid_route(xbar->route, xbar->rlen, 40, 20*2*1024);
    sleep(1);
    rc = find_the_packets(&nep, &nslot, &nport, 70, 85);
    if (rc > 0) {
      break;
    } else {
      printf("Retrying find...\n");
    }
  }

  /* If we never made it, just stop */
  if (rc == 0) {
    printf("Cannot find packets, giving up.\n");
    printf("WARNING - wire list will be only partial!\n");
    return -1;
  }

  /* Associate this host with its fiber port */
  associate_enclosure(nep, nslot, nport, xbar,
	mp->host[My_host_id].port[Hport].xbport);

  /*
   * Now, loop through all switches/slots/ports scouting unknown frontiers
   */
 restart:
  for (s=0; s<Nswitch; ++s) {
    ep = Switch + s;
    
    /* Find a fiber port that goes nowhere... */
    for (slot=1; slot<MAX_SLOT; ++slot) {
      sp = ep->slot[slot];
      if (sp == NULL || sp->num_fiber == 0) {
        continue;
      }

      for (p=0; p<sp->num_fiber; ++p) {
        fp = sp->fp + p;
        xp = fp->xp;
	sp2 = xp->ep->slot[xp->slotno];

	/* We need to find a fiber port whose underlying xbar known */
	xbar = sp2->xbar;
	if (xbar == NULL) {
	  continue;
	}
	xport = WPORT_TO_XPORT(xp->portno, sp2->port_delta);

	if (xbar->port[xport].conn_type != CONN_XBAR) {
	  continue;
	}

	/* OK, finally, this is the xbar on the other end of the fiber */
	xbar2 = xbar->port[xport].ptr.x;

	/*
	 * The test packet did not necessarily come in the same way we found
	 * the xbar, but we happened to save port the packet entered thru
	 * back when we were generating the routes.  Hence, we set xport2 as
	 * below instead of xport2 = xbar->port[xport].conn_port
	 */
	xport2 = xbar2->in_port;

	/* If not already mapped, find it! */
	if (xbar2->sp == NULL) {

	  for (retry=0; retry<3; ++retry) {
	    send_invalid_route(xbar2->route, xbar2->rlen, 40, 20*2*1024);
	    sleep(1);
	    rc = find_the_packets(&nep, &nslot, &nport, 70, 88);
	    if (rc > 0) {
	      break;
	    } else {
	      printf("Retrying find...\n");
	    }
	  }

	  /* If we never made it, just stop */
	  if (rc == 0) {
	    printf("Cannot find packets, giving up.\n");
	    printf("WARNING - wire list will be only partial!\n");
	    return -1;
	  }

	  /* Associate this new enclosure */
	  associate_enclosure(nep, nslot, nport, xbar2, xport2);
	  goto restart;
	}
      }
    }
  }

  printf("All done with enclosure mapping\n");
  return 0;
}

	  
void
output_wiring_list()
{
  struct enclosure *ep;
  struct slotdesc *sp;
  struct slotdesc *sp2;
  struct slotdesc *sp3;
  int e;
  int slot;
  int port;
	int hport;
  int mxp;
  struct fiberport *fp;
  struct fiberport *fp2;
  struct xbarport *xp2;
  struct xbar *xbar;
  FILE *f;

  if (Wirelist == NULL || strcmp(Wirelist, "-") == 0) {
    printf("Writing wirelist to stdout\n");
    f = stdout;
  } else {
    printf("Writing wirelist to %s\n", Wirelist);
    f = fopen(Wirelist, "w");
    assert(f != NULL);
  }

  for (e=0; e<Nswitch; ++e) {
    ep = Switch + e;

    for (slot=1; slot < MAX_SLOT; ++slot) {
      sp = ep->slot[slot];

      /* skip missing slots */
      if (sp == NULL) continue;

      fprintf(f, "\nEnclosure %s Slot %d serial %s model %s",
	ep->name, slot, sp->serialno, sp->model);
      if (sp->num_xp > 0) {
	fprintf(f, " %s", sp->xbar->name);
      }
      fprintf(f, "\n");

      for (port = 0; port < sp->num_fiber; ++port) {

	fp = sp->fp + port;
	fprintf(f, "Port %d ", port + sp->port_label_offset);

	/* Is our xbar known? */
	if (fp->xp == NULL || ep->slot[fp->xp->slotno]->xbar == NULL) {
	  fprintf(f, "????\n");

	} else {
	  sp2 = ep->slot[fp->xp->slotno];
	  xbar = sp2->xbar;
	  mxp = WPORT_TO_XPORT(fp->xp->portno, sp2->port_delta);

	  /* print the internal xbar:port for this port */
	  fprintf(f, "(%s:%d) ",
		  fp->xp->ep->slot[fp->xp->slotno]->xbar->name,
		  fp->xp->portno);

	  switch (xbar->port[mxp].conn_type) {
	  case CONN_NULL:
	    fprintf(f, "nothing\n");
	    break;
	  case CONN_HOST:
	    fprintf(f, "host %s, port ", xbar->port[mxp].ptr.hp->myhost->hostname);

		/* find the host port the xbar is connected to */
		for (hport = 0; hport < xbar->port[mxp].ptr.hp->myhost->nport; hport++) {
			if (&(xbar->port[mxp].ptr.hp->myhost->port[hport]) == xbar->port[mxp].ptr.hp) {
				fprintf(f, "%d\n", hport);
				break;
			}
		}
		assert(hport < xbar->port[mxp].ptr.hp->myhost->nport);
	    break;
	  case CONN_XBAR:
	    /* slot holding remote xbar */
	    sp3 = xbar->port[mxp].ptr.x->sp;

	    /* NULL sp3 means we never translated this xbar */
	    if (sp3 == NULL) {
	      fprintf(f, "untranslated_xbar\n");
	      break;
	    }

	    /* xbar port on that slot */
	    xp2 = sp3->xp + XPORT_TO_WPORT(xbar->port[mxp].conn_port,
			sp3->port_delta);

	    /* fiber port for that xbar port */
	    fp2 = xp2->fp;

	    /* print info about this fiber port */
	    fprintf(f, "switch %s slot %d port %d\n",
		  fp2->ep->name, fp2->slotno,
		  fp2->portno + fp2->ep->slot[fp2->slotno]->port_label_offset);
	    break;
	  }
	}
      }
    }
  }

  if (f != stdout) {
    fclose(f);
  }   
}

	  
void
output_wiring_dir(
    struct mapfile *mp
)
{
  struct enclosure *ep;
  struct slotdesc *sp;
  struct slotdesc *sp2;
  struct slotdesc *sp3;
  int e;
  int slot;
  int port;
  int hport;
  int host_id;
  int port_id;
  int mxp;
  struct fiberport *fp;
  struct fiberport *fp2;
  struct xbarport *xp2;
  struct xbar *xbar;
  char enc_fn[256];
  FILE *enc_fd;
  char host_fn[256];
  FILE *host_fd;
  char linecard_fn[256];
  FILE *linecard_fd;
  char link_fn[256];
  FILE *link_fd;
  char nic_fn[256];
  FILE *nic_fd;
  char host_str[256];
  char *port_id_str;
  char *last_colon;
  struct symtab *myhosts;

  if (Wiredir == NULL) {
    fprintf(stderr, "Wiring directory not specified\n");
    return;
  } else {
    fprintf(stderr, "Writing wirelist to directory %s\n", Wiredir);
  }

  /*
   * Create output files for FMA format wiring data
   */
  sprintf(enc_fn, "%s/%s", Wiredir, FMA_ENCLOSURES_FN);
  if ((enc_fd = fopen(enc_fn, "w")) == NULL) {
    fprintf(stderr, "Could not open %s for writting!\n", enc_fn);
    return;
  }
  fprintf(enc_fd, "%s\n", FMA_ENCLOSURES_LABELS_STR);
  fprintf(enc_fd, "%s\n", FMA_ENCLOSURES_TYPES_STR);
  
  sprintf(host_fn, "%s/%s", Wiredir, FMA_HOSTS_FN);
  if ((host_fd = fopen(host_fn, "w")) == NULL) {
    fprintf(stderr, "Could not open %s for writting!\n", host_fn);
    return;
  }
  fprintf(host_fd, "%s\n", FMA_HOSTS_LABELS_STR);
  fprintf(host_fd, "%s\n", FMA_HOSTS_TYPES_STR);
  
  sprintf(linecard_fn, "%s/%s", Wiredir, FMA_LINECARDS_FN);
  if ((linecard_fd = fopen(linecard_fn, "w")) == NULL) {
    fprintf(stderr, "Could not open %s for writting!\n", linecard_fn);
    return;
  }
  fprintf(linecard_fd, "%s\n", FMA_LINECARDS_LABELS_STR);
  fprintf(linecard_fd, "%s\n", FMA_LINECARDS_TYPES_STR);
  
  sprintf(link_fn, "%s/%s", Wiredir, FMA_LINKS_FN);
  if ((link_fd = fopen(link_fn, "w")) == NULL) {
    fprintf(stderr, "Could not open %s for writting!\n", link_fn);
    return;
  }
  fprintf(link_fd, "%s\n", FMA_LINKS_LABELS_STR);
  fprintf(link_fd, "%s\n", FMA_LINKS_TYPES_STR);
  
  sprintf(nic_fn, "%s/%s", Wiredir, FMA_NICS_FN);
  if ((nic_fd = fopen(nic_fn, "w")) == NULL) {
    fprintf(stderr, "Could not open %s for writting!\n", nic_fn);
    return;
  }
  fprintf(nic_fd, "%s\n", FMA_NICS_LABELS_STR);
  fprintf(nic_fd, "%s\n", FMA_NICS_TYPES_STR);
  
  for (e=0; e<Nswitch; ++e) {
    char *product_id;

    ep = Switch + e;

    /* determine enclosure model from backplane slots */
    switch (ep->num_bp_slots) {
    case 0:
      product_id = "M3-E16";
      break;
    case 2:
      product_id = "M3-E32";
      break;
    case 4:
      product_id = "M3-E64";
      break;
    case 8:
      product_id = "M3-E128";
      break;
    default:
      fprintf(stderr, "Bad number of backplane slots (%d) on %d\n",
	ep->name, ep->num_bp_slots);
      exit(1);
      break;
    }

    fprintf(enc_fd, "%s,%s\n", ep->name, product_id);

    for (slot=1; slot < MAX_SLOT; ++slot) {
      sp = ep->slot[slot];

      /* skip missing slots */
      if (sp == NULL) continue;

      /*jdjd - major hack to keep from reporting fake slots */
      if (slot > 32)
	continue;

      fprintf(linecard_fd, "%s,%d,%s,%s\n",
	ep->name, slot, sp->model, sp->serialno);

      for (port = 0; port < sp->num_fiber; ++port) {

	fp = sp->fp + port;

	/* Is our xbar known? */
	if (fp->xp == NULL || ep->slot[fp->xp->slotno]->xbar == NULL) {
	  fprintf(link_fd, "%s,%d,%d,%d,",
                  ep->name, slot, port + sp->port_label_offset, 0);
	  fprintf(link_fd, ",,,\n");

	} else {
	  sp2 = ep->slot[fp->xp->slotno];
	  xbar = sp2->xbar;
	  mxp = WPORT_TO_XPORT(fp->xp->portno, sp2->port_delta);

	  switch (xbar->port[mxp].conn_type) {
	  case CONN_NULL:
	    fprintf(link_fd, "%s,%d,%d,%d,,,,\n",
		    ep->name, slot, port + sp->port_label_offset, 0);
	    break;
	  case CONN_HOST:

	    strcpy(host_str, xbar->port[mxp].ptr.hp->myhost->hostname);
	
	    if ((last_colon = rindex(host_str, ':')) == NULL) {
	      /* Found the 1st Myrinet nic on the host. */
	      port_id_str = "0";	
	    } else {
	      /*
	       * Found the 2nd (or 3rd...) myrinet card in the host.
	       * Break out the hostname from the nic id (aka the
	       * "slot" to be consistent with switch terminology.
	       */
	      *last_colon = '\0';
	      port_id_str = last_colon + 1;
	
	      /* Look for special case where hostname has colons in it.
	       * The GM mapper assigns MX node MAC addresses to their names
	       * jdjdjd - this will break if MX nodes have multiple nics
	       */
	      if (rindex(host_str, ':') != NULL) {
		*last_colon = ':';
		port_id_str = "0";
	      }
	
	    }

	    /* If this is an 8E card, hosts are directly attached, no xcvr */
	    if (strcmp(sp->model, "M3-SW16-8E") == 0) {
	      fprintf(link_fd, "%s,%d,0,%d,",
		      ep->name, slot, port + sp->port_label_offset);
	    } else {
	      fprintf(link_fd, "%s,%d,%d,0,",
		      ep->name, slot, port + sp->port_label_offset);
	    }

	    fprintf(link_fd, "%s,%s,", host_str, port_id_str);
	    	
	    /* find the host port the xbar is connected to */
	    for (hport = 0; hport < xbar->port[mxp].ptr.hp->myhost->nport; hport++) {
		if (&(xbar->port[mxp].ptr.hp->myhost->port[hport]) == xbar->port[mxp].ptr.hp) {
			fprintf(link_fd, "%d,0\n", hport);
			break;
		}
	    }
	    assert(hport < xbar->port[mxp].ptr.hp->myhost->nport);
	    break;
	  case CONN_XBAR:
	    /* slot holding remote xbar */
	    sp3 = xbar->port[mxp].ptr.x->sp;

	    /* NULL sp3 means we never translated this xbar */
	    if (sp3 == NULL) {
	      break;
	    }

	    if (sp3 > sp) continue;

	    /* xbar port on that slot */
	    xp2 = sp3->xp + XPORT_TO_WPORT(xbar->port[mxp].conn_port,
			sp3->port_delta);

	    /* fiber port for that xbar port */
	    fp2 = xp2->fp;

	    /* print info about this fiber port */
	    fprintf(link_fd, "%s,%d,%d,%d,",
		    ep->name, slot, port + sp->port_label_offset, 0);
	    fprintf(link_fd, "%s,%d,%d,0\n", fp2->ep->name, fp2->slotno, 
		fp2->portno + fp2->ep->slot[fp2->slotno]->port_label_offset, 0);
	    break;
	  }
	}
      }
    }
  }

  /* Hosts and Nics */
  myhosts = symtab_init();

  for (host_id=0; host_id < mp->nhost; ++host_id) {
    strcpy(host_str, mp->host[host_id].hostname);

    if ((last_colon = rindex(host_str, ':')) == NULL) {
      /* Found the 1st Myrinet nic on the host. */
      port_id_str = "0";	
    } else {
      /*
       * Found the 2nd (or 3rd...) myrinet card in the host.
       * Break out the hostname from the nic id (aka the
       * "slot" to be consistent with switch terminology.
       */
      *last_colon = '\0';
      port_id_str = last_colon + 1;

      /* Look for special case where hostname has colons in it.
       * The GM mapper assigns MX node MAC addresses to their names
       * jdjdjd - this will break if MX nodes have multiple nics
       */
      if (rindex(host_str, ':') != NULL) {
	*last_colon = ':';
	port_id_str = "0";
      }

    }

    /*
     * Print out the hostname if it hasn't already been printed 
     */
    if (symtab_register(myhosts, host_str) != -1) {
      fprintf(host_fd, "%s,\n", host_str);
    }

    /* Determine product ID of NIC */
    {
      struct slotdesc *sp;
      char *product_id;

      /* slot we are connected to */
      sp = mp->host[host_id].port[0].xbar->sp;
      if (strcmp(sp->model, "M3-SW16-8E") == 0) {
        product_id = "XM_GIGE";

      } else if (mp->host[host_id].nport > 1) {
        product_id = "M3F-PCI64E-2";		/* a good guess */
      } else {
        product_id = "M3F-PCI64D-2";		/* a good guess */
      }

      fprintf(nic_fd, "%s,%s,%02x:%02x:%02x:%02x:%02x:%02x,,%s\n",
		host_str,
		port_id_str,
		mp->host[host_id].mac_addr[0],
		mp->host[host_id].mac_addr[1],
		mp->host[host_id].mac_addr[2],
		mp->host[host_id].mac_addr[3],
		mp->host[host_id].mac_addr[4],
		mp->host[host_id].mac_addr[5],
		product_id);
    }
  }


  fclose(enc_fd);
  fclose(link_fd);
  fclose(nic_fd);
  fclose(host_fd);
  fclose(linecard_fd);
}
